/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        PGPKeySet implementation

        $Id: pgpKeySet.c,v 1.42 2001/01/25 22:11:11 jeffc Exp $
____________________________________________________________________________*/

#include "pgpConfig.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <string.h>
#include <ctype.h>

#include "pgpTypes.h"
#include "pgpDebug.h"
#include "pgpKeyPriv.h"
#include "pgpMem.h"
#include "pgpTimeDate.h"
#include "pgpUsuals.h"
#include "pgpMemPool.h"
#include "pgpContext.h"
#include "pgpOptionListPriv.h"


	PGPBoolean
pgpKeySetIsMember(PGPKeyDBObj *obj, PGPKeySet *set)
{
	pgpa((
		pgpaPGPKeySetValid(set),
		pgpaPGPKeyDBObjValid(obj)));
		
	if ( !pgpKeyDBObjIsReal( obj ) )
		return( FALSE );
	
	return set->isMember(set, obj);
}

	PGPBoolean
PGPKeySetIsMember(PGPKeyDBObj *obj, PGPKeySet *set)
{
	pgpa((
		pgpaPGPKeySetValid(set),
		pgpaPGPKeyDBObjValid(obj)));
		
	if ( ! ( pgpKeySetIsValid( set ) && pgpKeyDBObjIsValid( obj ) ) )
		return( FALSE );

	pgpEnterBooleanFunction( FALSE );

	return pgpKeySetIsMember( obj, set );
}

#if 0
	PGPError
PGPUnionKeySets(PGPKeySetRef set1, PGPKeySetRef set2, PGPKeySetRef *newSet)
{
	PGPError	err	= kPGPError_NoErr;

	PGPValidatePtr( newSet );
	*newSet	= NULL;
	PGPValidateKeySet( set1 );
	PGPValidateKeySet( set2 );
	if (set1->keyDB != set2->keyDB)
		return kPGPError_KeyDBMismatch;

	pgpEnterPGPErrorFunction();

	if (set1->makeUnion == set2->makeUnion && IsntNull(set1->makeUnion))
		err = set1->makeUnion(set1, set2, newSet);
	else
		err = pgpGenericUnionOfKeySets(set1, set2, newSet);

	return err;
}
#endif

/* This does a union into the second set, overwriting it */
	PGPError
PGPAddKeys(PGPKeySetRef set, PGPKeySetRef dest)
{
	PGPKeySetRef	newSet;
	PGPKeySetRef	destCopy;
	PGPError		err	= kPGPError_NoErr;

	PGPValidateKeySet( set );
	PGPValidateKeySet( dest );

	pgpEnterPGPErrorFunction();

	if (set->keyDB != dest->keyDB)
		return kPGPError_KeyDBMismatch;

	/* Special case - union into root set makes no change */
	if( dest->keyDB->rootSet == dest )
		return kPGPError_NoErr;

	if( dest->refCount != 1 )
	{
		pgpDebugMsg( "PGPAddKeys can't overwrite dest if it is in use" );
		return kPGPError_BadParams;
	}

	/* Make an independent copy of dest for union to point at */
	err = pgpNewKeySetInternal(set->keyDB, &destCopy);
	if( IsPGPError( err ) )
		return err;
	destCopy->priv = dest->priv;
	destCopy->firstListInSet = dest->firstListInSet;
	destCopy->filterSubObjects = dest->filterSubObjects;
	destCopy->isMember = dest->isMember;
	destCopy->isEmpty = dest->isEmpty;
	destCopy->makeUnion = dest->makeUnion;
	destCopy->destroy = dest->destroy;

	if (set->makeUnion == destCopy->makeUnion && IsntNull(set->makeUnion))
		err = set->makeUnion(set, destCopy, &newSet);
	else
		err = pgpGenericUnionOfKeySets(set, destCopy, &newSet);

	/* Decrement ref count on destCopy, freeing it if appropriate */
	PGPFreeKeySet( destCopy );

	if( IsPGPError( err ) )
		return err;

	/* Overwrite contents of dest with newset */
	dest->priv				= newSet->priv;
	dest->firstListInSet	= newSet->firstListInSet;
	dest->filterSubObjects	= newSet->filterSubObjects;
	dest->isMember			= newSet->isMember;
	dest->makeUnion			= newSet->makeUnion;
	dest->destroy			= newSet->destroy;

	newSet->priv = NULL;
	PGPFreeKeySet( newSet );

	return kPGPError_NoErr;
}

/* Make a union of a single-key set and the dest, overwriting dest */
	PGPError
PGPAddKey(PGPKeyDBObjRef key, PGPKeySet *dest)
{
	PGPKeySetRef	oneset;
	PGPError		err	= kPGPError_NoErr;

	PGPValidateKeyDBObj( key );
	PGPValidateKeySet( dest );

	if( PGPPeekKeyDBObjKeyDB( key ) != dest->keyDB )
		return kPGPError_KeyDBMismatch;

	pgpEnterPGPErrorFunction();

	/* Special case - union into root set makes no change */
	if( dest->keyDB->rootSet == dest )
		return kPGPError_NoErr;

	/* Else create keyset from key and use addkeys */
	err = PGPNewOneKeySet( key, &oneset );
	if( IsPGPError( err ) )
		return err;
	err = PGPAddKeys( oneset, dest );
	PGPFreeKeySet( oneset );
	return err;
}


	PGPError
pgpNewKeySetInternal(
	PGPKeyDBRef		db,
	PGPKeySetRef *	newSet)
{
	PGPKeySet *		set;
	PGPContextRef	context	= PGPPeekKeyDBContext(db);

	pgpa(pgpaPGPKeyDBValid(db));

	*newSet = NULL;		/* In case there's an error */

	set = (PGPKeySet *)pgpContextMemAlloc(context,
							sizeof(PGPKeySet), kPGPMemoryMgrFlags_Clear);
	if (IsNull(set))
		return kPGPError_OutOfMemory;

	set->priv			= NULL;
	set->refCount		= 1;
	set->keyDB			= db;
	set->firstListInSet = NULL;
	set->magic			= kPGPKeySetMagic;

	set->prevSetInDB = NULL;
	set->nextSetInDB = db->firstSetInDB;
	if (set->nextSetInDB)
		set->nextSetInDB->prevSetInDB = set; 
	db->firstSetInDB = set;

	*newSet = set;
	return kPGPError_NoErr;
}

	static PGPBoolean
rootSetIsMember(PGPKeySet *set, PGPKeyDBObjRef key)
{
	pgpa((
		pgpaPGPKeySetValid(set),
		pgpaAddrValid(key, char)));	/* XXX use better align check */

	return PGPPeekKeyDBObjKeyDB(key) == set->keyDB;
}

	static PGPBoolean
rootSetIsEmpty(PGPKeySet *set)
{
	pgpa((pgpaPGPKeySetValid(set)));

	return IsNull( set->keyDB->firstKeyInDB );
}

	static PGPError
rootSetMakeUnion(PGPKeySetRef set1, PGPKeySetRef set2, PGPKeySetRef *newSet)
{
	(void)set2;	/* Avoid warning */

	PGPIncKeySetRefCount(set1);
	*newSet = set1;
	return kPGPError_NoErr;
}

	static void
rootSetDestroy(PGPKeySet *set)
{
	(void)set;	/* Avoid warning */
}


	PGPKeySet *
pgpRootSet(PGPKeyDB *		db)
{
	PGPKeySet *		set;

	(void)pgpNewKeySetInternal(db, &set);
	if (IsntNull(set))
	{
		set->isMember = rootSetIsMember;
		set->isEmpty = rootSetIsEmpty;
		set->makeUnion = rootSetMakeUnion;
		set->destroy = rootSetDestroy;
	}
	return set;
}

	PGPError
PGPNewKeySet( PGPKeyDB *db, PGPKeySetRef *pset )
{
	PGPKeySet		*set;

	PGPValidateKeyDB( db );
	PGPValidatePtr( pset );
	*pset = NULL;

	pgpEnterPGPErrorFunction();

	set = pgpRootSet( db );
	if( set == NULL )
		return kPGPError_OutOfMemory;
	*pset = set;
	return kPGPError_NoErr;
}
	


	PGPError
PGPIncKeySetRefCount(PGPKeySetRef keys)
{
	PGPValidateKeySet( keys );

	pgpEnterPGPErrorFunction();

	/* Root set doesn't get refcounted */
	if( keys->keyDB->rootSet != keys )
		keys->refCount++;
	
	return( kPGPError_NoErr );
}


/* Add all the objects in the first key set into the second keydb. */
/* This may be run on front end or on back end */
	PGPError
pgpCopyKeys_internal (
	PGPKeySetRef	keysToAdd,
	PGPKeyDBRef		destdb,
	PGPKeySetRef	*destKeySet )
{
	PGPKeyIter	   *iter = NULL;
	PGPKeyDBObj	   *obj;
	PGPKeyDBObj	   *newobj;
	PGPKeySet	   *newset = NULL;
	PGPKeySet	   *oneset;
	PGPUInt32		nkeys;
	PGPError		err = kPGPError_NoErr;

	/* Clear db add cache */
	pgpClearMemory( destdb->oldobjs, sizeof(destdb->oldobjs) );
	pgpClearMemory( destdb->newobjs, sizeof(destdb->newobjs) );

	if( IsntNull( destKeySet ) )
	{
		PGPCountKeysInKeyDB( destdb, &nkeys );
		if( nkeys != 0 )
		{
			err = PGPNewEmptyKeySet( destdb, &newset );
			if ( IsPGPError( err ) )
				goto error;
		}
	}
	
	/*
	 * Shortcut if we can flatten source key set, do the add in one
	 * call (more calls needed to resync though).
	 */
	if( pgpFrontEndKeyDB( destdb ) &&
		pgpKeySetIsFlattenable( keysToAdd ) )
	{
		PGPUInt32 *			keylist;
		PGPUInt32 *			newkeylist = NULL;
		PGPSize				keylistsize;
		PGPSize				newkeylistsize;
		PGPUInt32			numKeys;
		PGPUInt32 *			keyArray;
		PGPSize				keyArraySize;
		PGPBoolean			neednewkeylist;
		PGPBoolean			allNewKeys;

		if( IsntNull( newset ) )
			PGPFreeKeySet( newset );
		/* Always need new key list so we can refresh returned keys */
		neednewkeylist = TRUE; /*IsntNull( destKeySet ) && nkeys != 0;*/
		err = pgpKeySetFlatten( keysToAdd, &keylist, &keylistsize );
		if( IsPGPError( err ) )
			return err;
		err = pgpCopyKeys_back( PGPPeekKeyDBContext(destdb),
								PGPPeekKeySetKeyDB(keysToAdd)->id, destdb->id,
								keylist, keylistsize, neednewkeylist,
								&newkeylist, &newkeylistsize );
		if( IsPGPError( err ) )
			return err;
		/* Update front-end copy of destdb */
		err = pgpKeyDBArray_back( PGPPeekKeyDBContext(destdb), destdb->id,
								  &numKeys, &keyArray, &keyArraySize );
		if( IsPGPError( err ) )
		{
			if( IsntNull( newkeylist ) )
				PGPFreeData( newkeylist );
			return err;
		}
		allNewKeys = IsNull(destdb->firstKeyInDB);
		err = pgpAddFromKeyArray( destdb, NULL, keyArray, numKeys, allNewKeys);
		if( IsntNull( keyArray ) )
			PGPFreeData( keyArray );
		if( IsPGPError( err ) )
		{
			if( IsntNull( newkeylist ) )
				PGPFreeData( newkeylist );
			return err;
		}
		/* Set destKeySet if requested */
		if( IsntNull( destKeySet ) )
		{
			if( nkeys == 0 )
			{
				/* Want root key set */
				err = PGPNewKeySet( destdb, destKeySet );
			} else {
				err = pgpKeySetUnflatten( destdb, newkeylist,
										  newkeylistsize, destKeySet );
			}
		}
		/* Frees list */
		pgpKeyRefreshFromList( destdb, newkeylist, newkeylistsize );
		/* No need to call pgpKeyDBChanged, was called in AddFromKeyArray */
		return err;
	}

	err = PGPNewKeyIterFromKeySet( keysToAdd, &iter );
	if ( IsPGPError( err ) )
		goto error;

	while( IsntPGPError( pgpKeyIterNextObject( iter, &obj ) ) ) {
		err = pgpKeyDBAddObject( destdb, obj, &newobj );
		if ( IsPGPError( err ) )
			goto error;
		if( pgpFrontEndKeyDB( destdb ) )
			pgpKeyDBObjRefresh( newobj, FALSE );
		if( IsntNull( newset ) && nkeys != 0 )
		{
			err = PGPNewOneKeySet( newobj, &oneset );
			if ( IsPGPError( err ) )
				goto error;
			err = PGPAddKeys( oneset, newset );
			PGPFreeKeySet( oneset );
			if ( IsPGPError( err ) )
				goto error;
		}
	}

	PGPFreeKeyIter( iter );
	iter = NULL;

	if( pgpFrontEndKeyDB( destdb ) )
	{
		PGPUInt32		numKeys;
		PGPUInt32 *		keyArray;
		PGPSize			keyArraySize;

		/* Refresh from back end in case got some dummy keys */
		err = pgpKeyDBArray_back( PGPPeekKeyDBContext(destdb), destdb->id,
								  &numKeys, &keyArray, &keyArraySize );
		if( IsPGPError( err ) )
			goto error;
		err = pgpAddFromKeyArray( destdb, NULL, keyArray, numKeys, FALSE );
		if( IsntNull( keyArray ) )
			PGPFreeData( keyArray );
		if( IsPGPError( err ) )
			goto error;
	}


	if( IsntNull( destKeySet ) )
	{
		if( nkeys == 0 )
		{
			err = PGPNewKeySet( destdb, destKeySet );
		}  else {
			*destKeySet = newset;
			newset = NULL;
		}
	}
	pgpKeyDBChanged( destdb, TRUE );

error:
	if( IsntNull(iter) )
		PGPFreeKeyIter (iter);
	if( IsntNull(newset) )
		PGPFreeKeySet (newset);
		
	return err;
}


/* Add all the objects in the first key set into the second keydb. */
	PGPError
PGPCopyKeys (
	PGPKeySetRef	keysToAdd,
	PGPKeyDBRef		destdb,
	PGPKeySetRef	*destKeySet )
{
	if( IsntNull( destKeySet ) )
		*destKeySet = kInvalidPGPKeySetRef;
		
	PGPValidateKeyDB( destdb );
	PGPValidateKeySet( keysToAdd );

	pgpEnterPGPErrorFunction();

	return pgpCopyKeys_internal( keysToAdd, destdb, destKeySet );
}


/* Like PGPCopyKeys, but just one key */
	PGPError
PGPCopyKeyDBObj (
	PGPKeyDBObjRef	obj,
	PGPKeyDBRef		keydb,
	PGPKeyDBObjRef	*destObj)
{
	PGPKeySetRef	oneset;
	PGPKeySetRef	newset;
	PGPError		err	= kPGPError_NoErr;

	if( IsntNull( destObj ) )
		*destObj = kInvalidPGPKeyDBObjRef;
		
	PGPValidateKeyDBObj( obj );
	PGPValidateKeyDB( keydb );

	pgpEnterPGPErrorFunction();

	err = PGPNewOneKeySet( obj, &oneset );
	if( IsPGPError( err ) )
		return err;
	err = PGPCopyKeys( oneset, keydb, (IsntNull(destObj)?&newset:NULL) );
	PGPFreeKeySet( oneset );

	if( IsntPGPError(err) && IsntNull( destObj ) )
	{
		*destObj = pgpFirstKeyInKeySet( newset );
		PGPFreeKeySet( newset );
	}

	return err;
}



/* Remove all objects in the set from its keydb */
	PGPError
PGPDeleteKeys (
	PGPKeySetRef	keysToRemove)
{
	PGPKeyIter			*iter = NULL;
	PGPKeyDB			*db;
	PGPKeyDBObj			*obj;
	PGPError			err = kPGPError_NoErr;

	PGPValidateKeySet( keysToRemove );

	pgpEnterPGPErrorFunction();

	db = PGPPeekKeySetKeyDB( keysToRemove );

	err = PGPNewKeyIterFromKeySet( keysToRemove, &iter );
	if ( IsPGPError( err ) )
		goto error;

	while( IsntPGPError( pgpKeyIterNextObject( iter, &obj ) ) ) {
		/* May have been deleted when parent was */
		if( !pgpKeyDBObjIsDeleted( obj ) )
		{
			err = pgpKeyDBRemoveObject( db, obj );
			if ( IsPGPError( err ) )
				goto error;
		}
	}

	pgpKeyDBChanged( db, TRUE );
	PGPFreeKeyIter( iter );
	iter = NULL;

error:
	if (iter)
		PGPFreeKeyIter (iter);
	return err;
}

	PGPError
pgpFreeKeySet(PGPKeySet *keys, PGPBoolean bForce)
{

	/* Root set doesn't get refcounted */
	if( keys->keyDB->rootSet != keys )
		keys->refCount--;

	if (keys->refCount <= 0  ||  bForce)
	{
		(*keys->destroy)(keys);
		keys->magic	= ~keys->magic;	/* mark as invalid */
		
		if (keys->prevSetInDB)
			keys->prevSetInDB->nextSetInDB = keys->nextSetInDB;
		else
			keys->keyDB->firstSetInDB = keys->nextSetInDB;
		if (keys->nextSetInDB)
			keys->nextSetInDB->prevSetInDB = keys->prevSetInDB;

		pgpAssert(keys->firstListInSet == NULL);

		PGPFreeData( keys);
	}
	
	return( kPGPError_NoErr );
}

	PGPError
PGPFreeKeySet(PGPKeySet *keys)
{
	PGPValidateKeySet( keys );
	
	pgpEnterPGPErrorFunction();

	return pgpFreeKeySet( keys, FALSE );
}



	PGPError
PGPCountKeys(
	PGPKeySetRef	keys,
	PGPUInt32 *		numKeys )
{
	PGPKeyDBObj *		key;
	PGPUInt32		count = 0;
	PGPError		err	= kPGPError_NoErr;
	
	PGPValidatePtr( numKeys );
	*numKeys	= 0;
	PGPValidateKeySet( keys );

	pgpEnterPGPErrorFunction();

	for (key = keys->keyDB->firstKeyInDB; key; key = key->next)
	{
		pgpa(pgpaPGPKeyValid(key));
		if (pgpKeySetIsMember(key, keys))
			count++;
	}
	
	*numKeys = count;

	return( err );
}


	PGPBoolean
pgpKeySetIsEmpty(
	PGPKeySetRef	keys )
{
	pgpa(pgpaPGPKeySetValid(keys));
	return keys->isEmpty( keys );
}

	PGPKeyDBObj *
pgpFirstKeyInKeySet( PGPKeySet const * keySet )
{
	PGPKeyDBObj *	key;
	PGPKeyDB *		kdb;

	kdb = keySet->keyDB;

	for( key = kdb->firstKeyInDB; IsntNull(key); key = key->next )
		if( pgpKeySetIsMember( key, (PGPKeySet *)keySet ) )
			break;

	return key;
}


/* Sort keylist by keyid, only in back end for efficiency */
	static int
sCompareKeylistEntry( const void *ent1, const void *ent2 )
{
	PGPKeyDBObj *		obj1;
	PGPKeyDBObj *		obj2;
	PGPKeyID			keyid1, keyid2;

	obj1 = *(PGPKeyDBObj **) ent1;
	obj2 = *(PGPKeyDBObj **) ent2;

	pgpAssert( !pgpFrontEndKey( obj1 ) );
	pgpAssert( !pgpFrontEndKey( obj2 ) );

	PGPGetKeyID( obj1, &keyid1 );
	PGPGetKeyID( obj2, &keyid2 );
	
	return PGPCompareKeyIDs( &keyid1, &keyid2 );
}


	static void
sSortKeylist( PGPUInt32 *keylist, PGPSize keylistcount )
{
	qsort( keylist, keylistcount, sizeof(PGPUInt32),
		   sCompareKeylistEntry );
}


/* True if keyset is such that all keys contain all sub objects */
	PGPBoolean
pgpKeySetIsFlattenable( PGPKeySet const *set )
{
	PGPKeyIter *iter;
	PGPKeyDBObj *obj;
	PGPBoolean keyinset = FALSE;
	PGPError err;

	if( IsNull( set ) )
		return TRUE;

	/* Iterate over all keys in underlying keydb */
	if( IsPGPError( err = PGPNewKeyIterFromKeyDB(
							PGPPeekKeySetKeyDB((PGPKeySet*)set), &iter ) ) )
		return FALSE;

	/* If find an obj not in set when key was in set, can't do it */
	while( IsntPGPError( pgpKeyIterNextObject( iter, &obj ) ) )
	{
		/* Skip dummy objects */
		if( !pgpKeyDBObjIsReal( obj ) )
			continue;
		/* Skip if we find a key that is not a member */
		if( OBJISTOPKEY( obj ) )
		{
			keyinset = pgpKeySetIsMember( obj, (PGPKeySet *)set );
		} else {
			if( keyinset && !pgpKeySetIsMember( obj, (PGPKeySet *)set ) )
			{
				PGPFreeKeyIter( iter );
				return FALSE;
			}
		}
	}
	PGPFreeKeyIter( iter );
	return TRUE;
}



/* Create a list of all keyids of keys in key set */
	PGPError
pgpKeySetFlatten( PGPKeySet const *set, PGPUInt32 **keylist,
	PGPSize *keylistsize )
{
	PGPKeyIter *iter;
	PGPUInt32 *list;
	PGPSize size;
	PGPUInt32 nkeys;
	PGPUInt32 i;
	PGPKeyDBObjRef key;
	PGPContextRef context;
	PGPBoolean front;
	PGPError err;

	if( IsNull( set ) )
	{
		*keylist = NULL;
		*keylistsize = 0;
		return kPGPError_NoErr;
	}

	front = pgpFrontEndKeyDB( set->keyDB );

	PGPCountKeys( (PGPKeySet *)set, &nkeys );
	size = nkeys * sizeof(PGPUInt32);

	context = PGPPeekKeySetContext( (PGPKeySet *)set );

	list = (PGPUInt32 *)pgpContextMemAlloc( context, size, 0 );

	if( IsPGPError( err = PGPNewKeyIterFromKeySet( (PGPKeySet *)set, &iter )))
	{
		PGPFreeData( list );
		return err;
	}

	i = 0;
	while( IsntPGPError( PGPKeyIterNextKeyDBObj( iter, kPGPKeyDBObjType_Key,
												 &key ) ) )
	{
		list[i++] = front ? pgpKeyDBObjID( key ) : (PGPUInt32)key;
	}
	PGPFreeKeyIter( iter );

	pgpAssert( i == nkeys );

	/* Sort lists on back end */
	if( !front )
		sSortKeylist( list, nkeys );

	*keylist = list;
	*keylistsize = size;
	return kPGPError_NoErr;
}

/* Flatten and free the keyset */
	PGPError
pgpKeySetFlattenFree( PGPKeySet *set, PGPUInt32 **keylist,
	PGPSize *keylistsize )
{
	PGPError err;
	err = pgpKeySetFlatten( set, keylist, keylistsize );
	if( IsntNull( set ) )
		PGPFreeKeySet( set );
	return err;
}


/* Create a new keyset from a keylist, which is already sorted by keyid */
	PGPError
pgpKeySetUnflatten( PGPKeyDB *db, PGPUInt32 *keylist, PGPSize keylistsize,
	PGPKeySetRef *pset)
{
	PGPKeySetRef set;
	PGPBoolean front;
	PGPUInt32 keylistcount;
	PGPError err = kPGPError_NoErr;

	keylistcount = keylistsize / sizeof(PGPUInt32);

	front = pgpFrontEndKeyDB( db );

	/* Sort lists on back end */
	if( !front )
	{
		sSortKeylist( keylist, keylistcount );
	} else {
		/* Translate backend key_ids into key pointers */
		PGPUInt32 obj;
		PGPUInt32 i;
		for( i=0; i<keylistcount; ++i )
		{
			PGPFindNode( db->idToObj, keylist[i], (PGPUserValue *)&obj );
			keylist[i] = obj;
		}
	}

	if( keylistcount == 0 )
		err = PGPNewEmptyKeySet( db, &set );
	else
		err = pgpNewKeyListSet( db, (PGPKeyDBObjRef *)keylist, keylistcount,
								&set );

	if( IsntPGPError( err ) )
		*pset = set;
	return err;
}

/* Also free keylist */
	PGPError
pgpKeySetUnflattenFree( PGPKeyDB *db, PGPUInt32 *keylist, PGPSize keylistsize,
	PGPKeySetRef *pset)
{
	PGPError err;
	err = pgpKeySetUnflatten( db, keylist, keylistsize, pset );
	if( IsntNull( keylist ) )
		PGPFreeData( keylist );
	return err;
}

/* Refresh key objs in the keyset, i.e. mark them as being obsolete */
	PGPError
pgpKeySetRefreshFree( PGPKeySet *set )
{
	PGPKeyIter *iter;
	PGPKeyDBObjRef key;
	PGPError err;
	
	if( IsPGPError( err = PGPNewKeyIterFromKeySet( (PGPKeySet *)set, &iter )))
	{
		PGPFreeKeySet( set );
		return err;
	}

	while( IsntPGPError( PGPKeyIterNextKeyDBObj( iter, kPGPKeyDBObjType_Key,
												 &key ) ) )
	{
		(void) pgpKeyDBObjRefresh( key, TRUE );
	}
	PGPFreeKeyIter( iter );
	PGPFreeKeySet( set );
	
	return kPGPError_NoErr;
}

	PGPBoolean
pgpKeySetIsValid( PGPKeySet const *	keySet)
{
	return( IsntNull( keySet ) && keySet->magic == kPGPKeySetMagic );
}

#if PGP_DEBUG	/* [ */
	PGPBoolean
pgpaInternalPGPKeySetValid(
	pgpaCallPrefixDef,
	PGPKeySet const *	keySet,
	char const *		varName)
{
	pgpaAddrValid(keySet, PGPKeySet);
	pgpaFailIf(keySet->refCount <= 0, (pgpaFmtPrefix, "refCount <= 0"));
	pgpaFmtMsg((pgpaFmtPrefix,
			"pgpaPGPKeySetValid failed on %s (%p)", varName, keySet));

	return pgpaFailed;
}
#endif /* ] PGP_DEBUG */


/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
